﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Threading;

namespace SWAT_Office_App
{
    public struct DriveEntry
    {
        public DateTime time;   // Time of insertion
        public string drive;    // Drive letter of the drive
        public string label;    // Label of the drive
        public string size;     // Size of the drive in bytes (I think)
        public string owner;    // Owner to be specified (optional)
        public string dock;     // Dock number as choosen
    }
    public partial class DriveLogger_Form : Form
    {
        private static List<DriveEntry> driveEntryList= new List<DriveEntry>();
        private static string logLocation = "DriveLogger Log.txt";
        public static bool instanceAlreadyRunning = false;

        public DriveLogger_Form()
        {
            // Disable thread safe checking
            CheckForIllegalCrossThreadCalls = false;

            driveEntryList.Clear();
            InitializeComponent();

            this.listView_Drives.AfterLabelEdit += new LabelEditEventHandler(listView_Drives_AfterLabelEdit);
            this.KeyPreview = true;
            this.KeyDown += new KeyEventHandler(DriveLogger_Form_KeyDown);
            this.FormClosed += new FormClosedEventHandler(DriveLogger_Form_FormClosed);

            using (StreamWriter sw = File.AppendText(logLocation))
            {
                sw.WriteLine("-- New Session Started --");
                sw.WriteLine("Initializing form details");
            }

            // Adds columns to the listview
            this.listView_Drives.Columns.Add("Owner", 145, HorizontalAlignment.Left);
            this.listView_Drives.Columns.Add("Dock", 60, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Time", 130, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Drive", 40, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Label", 109, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Size", 60, HorizontalAlignment.Center);

            // Initializes and starts a backgroundworker thread for monitoring for new drives
            BackgroundWorker bgWorker = new BackgroundWorker();
            instanceAlreadyRunning = true;
            bgWorker.DoWork += new DoWorkEventHandler(bgWorker_DoWork);
            bgWorker.RunWorkerAsync();
        }
        void bgWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            BackgroundWorker worker = sender as BackgroundWorker;
            // Compares a list of drives detected previously with the a list of current
            // drives on the computer. If a change in length is detected, a drive is either
            // added or removed so the drive in question is found and operated upon
            List<string> drivesPreviouslyDetectedList = new List<string>();
            DriveInfo[] newlyDetectedDrives = DriveInfo.GetDrives();

            // Resets both lists to hold the same drives
            drivesPreviouslyDetectedList.Clear();
            foreach (DriveInfo drive in newlyDetectedDrives)
                drivesPreviouslyDetectedList.Add(drive.Name);

            while (true)
            {
                // Cancels thread if DriveLogger form is closed
                if (!instanceAlreadyRunning)
                {
                    //MessageBox.Show("Thread Stopped");
                    e.Cancel = true;
                    break;
                }
                // Pulls new list of current drives on computer
                newlyDetectedDrives = DriveInfo.GetDrives();
                if (newlyDetectedDrives.Length > drivesPreviouslyDetectedList.Count)
                {
                    // Applies if a drive is attached to the computer
                    // Goes through each list and finds the drive that was recently attached
                    bool newDrivesDetected = false;
                    string driveOwner = "";
                    string driveDock = "";
                    foreach (DriveInfo drive in newlyDetectedDrives)
                    {
                        // Skips the drive if it is in the list of drives to ignore
                        if (Settings_Form.driveLoggerDrivesToIgnore.Contains(drive.Name))
                            continue;

                        // Check for non-matching entries
                        if (!drivesPreviouslyDetectedList.Contains(drive.Name))
                        {
                            // Creates and populates a new DriveEntry
                            DriveEntry newEntry = new DriveEntry();

                            //if (newDrivesDetected == false)
                            //{
                            //    // First entry detected is prompted for dock label and owner name
                            //    newDrivesDetected = true;
                            //    DriveLogger_LabelPrompt_Form newPrompt = new DriveLogger_LabelPrompt_Form();
                            //    newPrompt.ShowDialog();
                            //    newEntry.owner = newPrompt.driveOwner;
                            //    driveOwner = newPrompt.driveOwner;
                            //    newEntry.dock = newPrompt.driveDock;
                            //    driveDock = newPrompt.driveDock;
                            //}
                            //else
                            //{
                            //    // Subsequent entries use the previously entered dock label and owner
                            //    newDrivesDetected = true;
                            //    newEntry.owner = driveOwner;
                            //    newEntry.dock = driveDock;
                            //}
                            
                            // Sets other drive details for the drive
                            newEntry.time = DateTime.Now;
                            newEntry.drive = drive.Name;

                            DriveInfo tempDrive = null;
                            DriveInfo[] allDrives = DriveInfo.GetDrives();
                            foreach (DriveInfo driveSearch in allDrives)
                            {
                                if (driveSearch.IsReady)
                                {
                                    if (driveSearch.Name == newEntry.drive)
                                    {
                                        tempDrive = driveSearch;
                                        break;
                                    }
                                }
                            }
                            newEntry.label = tempDrive.VolumeLabel;

                            // Determines the size of the attached drive
                            if ((tempDrive.TotalSize / 1073741824) > 0)
                                newEntry.size = (tempDrive.TotalSize / 1073741824).ToString() + " GB";
                            else
                                newEntry.size = (tempDrive.TotalSize / 1048576).ToString() + " MB";

                            // Adds the new DriveEntry into driveList
                            driveEntryList.Add(newEntry);

                            using (StreamWriter sw = File.AppendText(logLocation))
                            {
                                sw.WriteLine("Drive Attached -- [" + newEntry.time.ToString() + "]\t\"" + newEntry.owner + "\"\t" + newEntry.drive + "\t\"" + newEntry.label + "\"\t" + newEntry.size);
                            }

                            paintDriveListbox();
                        }
                    }
                    // Clears and refreshes the drivesPreviouslyDetected list
                    drivesPreviouslyDetectedList.Clear();
                    foreach (DriveInfo drive in newlyDetectedDrives)
                    {
                        drivesPreviouslyDetectedList.Add(drive.Name);
                    }
                }
                if (newlyDetectedDrives.Length < drivesPreviouslyDetectedList.Count)
                {
                    // Applies if a drive is removed to the computer
                    // Goes through each list and finds the drive that was recently removed
                    bool removedDriveFound = false;
                    foreach (string str in drivesPreviouslyDetectedList)
                    {
                        // Skips the drive if it is in the list of drives to ignore
                        if (Settings_Form.driveLoggerDrivesToIgnore.Contains(str))
                            continue;
                        
                        removedDriveFound = false;
                        // Loop here checks for non-matching entries in the two lists
                        foreach (DriveInfo drive in newlyDetectedDrives)
                        {
                            // If entries match, drive was not removed
                            if (str == drive.Name)
                            {
                                removedDriveFound = true;
                                break;
                            }
                        }
                        // If list mismatch is detected, remove from driveEntryList
                        if (removedDriveFound == false)
                        {
                            // Removes drive from driveList
                            foreach (DriveEntry entry in driveEntryList)
                            {
                                if (str == entry.drive)
                                {
                                    driveEntryList.Remove(entry);
                                    // Log the removal of the drive
                                    using (StreamWriter sw = File.AppendText(logLocation))
                                    {
                                        sw.WriteLine("Drive Removed --  [" + entry.time.ToString() + "]\t" + entry.drive + "\t\"" + entry.label + "\"\t" + entry.size);
                                    }
                                    break;
                                }
                            }
                            paintDriveListbox();
                        }
                    }
                    // Clears and refreshes the drivesPreviouslyDetected list
                    drivesPreviouslyDetectedList.Clear();
                    foreach (DriveInfo drive in newlyDetectedDrives)
                    {
                        drivesPreviouslyDetectedList.Add(drive.Name);
                    }
                }

                // Sleeps the thread for a specified amount of time
                Thread.Sleep(Settings_Form.driveLoggerRefreshInterval);
            }
        }
        void DriveLogger_Form_KeyDown(object sender, KeyEventArgs e)
        {
            // Monitors for keypresses for the about box and refresh
            switch (e.KeyCode)
            {
                case Keys.OemQuestion:
                    About_Box_Form window = new About_Box_Form();
                    window.ShowDialog();
                    break;
                case Keys.F5:
                    refreshDrives();
                    break;
            }
        }
        void listView_Drives_AfterLabelEdit(object sender, LabelEditEventArgs e)
        {
            // Logs the new label if it is changed
            if (e.Label != null)
            {
                using (StreamWriter sw = File.AppendText(logLocation))
                {
                    ListViewItem entry = listView_Drives.Items[e.Item];

                    sw.WriteLine("Label \"" + e.Label + "\" added to drive " + entry.SubItems[3].Text);
                }
            }
        }
        void DriveLogger_Form_FormClosed(object sender, FormClosedEventArgs e)
        {
            // Resets variable so form can be opened again
            instanceAlreadyRunning = false;
        }
        private void paintDriveListbox()
        {
            // Updates the listview
            this.listView_Drives.BeginUpdate();

            this.listView_Drives.Items.Clear();

            // Adds each entry from the driveList into the listView
            foreach (DriveEntry entry in driveEntryList)
            {
                if (entry.owner != "System Reserved")
                {
                    ListViewItem item = new ListViewItem();
                    item.Text = entry.owner;
                    ListViewItem.ListViewSubItem subDock = new ListViewItem.ListViewSubItem();
                    subDock.Text = entry.dock;
                    item.SubItems.Add(subDock);
                    ListViewItem.ListViewSubItem subTime = new ListViewItem.ListViewSubItem();
                    subTime.Text = entry.time.ToString();
                    item.SubItems.Add(subTime);
                    ListViewItem.ListViewSubItem subDrive = new ListViewItem.ListViewSubItem();
                    subDrive.Text = entry.drive;
                    item.SubItems.Add(subDrive);
                    ListViewItem.ListViewSubItem subLabel = new ListViewItem.ListViewSubItem();
                    subLabel.Text = entry.label;
                    item.SubItems.Add(subLabel);
                    ListViewItem.ListViewSubItem subSize = new ListViewItem.ListViewSubItem();
                    subSize.Text = entry.size;
                    item.SubItems.Add(subSize);

                    this.listView_Drives.Items.Add(item);
                }
            }

            this.listView_Drives.EndUpdate();
        }
        private void refreshDrives()
        {
            // Checks list of drive entries to see if any drives have been removed
            bool removedDriveFound;
            List<DriveEntry> drivesToRemove = new List<DriveEntry>();
            // Checks each entry in driveList to see if matching drive is found on list
            // of current drives. Removes entry from driveList if drive is not found.
            DriveInfo[] currentDrives = DriveInfo.GetDrives();
            for (int i = 0; i < driveEntryList.Count; i++)
            {
                DriveEntry storedDrive = driveEntryList[i];
                removedDriveFound = false;
                // Loop here checks for non-matching entries in the two lists
                foreach (DriveInfo currentDrive in currentDrives)
                {
                    // If entries match, drive was not removed
                    if (storedDrive.drive == currentDrive.Name)
                    {
                        removedDriveFound = true;
                        break;
                    }
                }
                // If list mismatch is detected, remove from driveList
                if (removedDriveFound == false)
                {
                    drivesToRemove.Add(storedDrive);
                }
            }
            // Removes drive from driveList
            foreach (DriveEntry entry in drivesToRemove)
            {
                // Removes drive from driveList
                foreach (DriveEntry entry2 in driveEntryList)
                {
                    if (entry.drive == entry2.drive)
                    {
                        driveEntryList.Remove(entry);
                        break;
                    }
                }
            }

            paintDriveListbox();
        }
    }
}
